import Foundation

/// Standardized health data input structure for actuarial calculations
public struct HealthMetrics: Hashable {
    public let restingHeartRate: Int?
    public let hrv: Double? // Heart Rate Variability in ms
    public let vo2Max: Double? // VO2 Max in ml/kg/min
    public let bmi: Double?
    public let stepCount: Int?
    public let sleepHours: Double?
    
    // Additional properties needed by ActuarialModel
    public let smokingStatus: SmokingStatus
    public let exerciseFrequency: ExerciseFrequency
    
    /// Convenience initializer with Double for resting heart rate to support existing code
    public init(restingHeartRate: Double?, hrv: Double? = nil, vo2Max: Double? = nil, bmi: Double? = nil, stepCount: Int? = nil, sleepHours: Double? = nil, smokingStatus: SmokingStatus = .never, exerciseFrequency: ExerciseFrequency = .none) {
        self.restingHeartRate = restingHeartRate != nil ? Int(restingHeartRate!) : nil
        self.hrv = hrv
        self.vo2Max = vo2Max
        self.bmi = bmi
        self.stepCount = stepCount
        self.sleepHours = sleepHours
        self.smokingStatus = smokingStatus
        self.exerciseFrequency = exerciseFrequency
    }
    
    /// Primary initializer with Int for resting heart rate
    public init(restingHeartRate: Int?, hrv: Double? = nil, vo2Max: Double? = nil, bmi: Double? = nil, stepCount: Int? = nil, sleepHours: Double? = nil, smokingStatus: SmokingStatus = .never, exerciseFrequency: ExerciseFrequency = .none) {
        self.restingHeartRate = restingHeartRate
        self.hrv = hrv
        self.vo2Max = vo2Max
        self.bmi = bmi
        self.stepCount = stepCount
        self.sleepHours = sleepHours
        self.smokingStatus = smokingStatus
        self.exerciseFrequency = exerciseFrequency
    }
    
    /// Validates that health metrics are within reasonable bounds
    public func isValid() -> Bool {
        if let rhr = restingHeartRate, !(40...120).contains(rhr) { return false }
        if let hrv = hrv, !(10...100).contains(hrv) { return false }
        if let vo2 = vo2Max, !(10...80).contains(vo2) { return false }
        if let bmi = bmi, !(12...60).contains(bmi) { return false }
        if let steps = stepCount, steps < 0 { return false }
        if let sleep = sleepHours, !(2...12).contains(sleep) { return false }
        return true
    }
    
    /// Extended validation with error throwing for actuarial use
    public func validate() throws {
        if let rhr = restingHeartRate {
            guard rhr >= 30 && rhr <= 220 else {
                throw ActuarialError.invalidHealthMetric("Resting heart rate must be between 30-220 bpm")
            }
        }
        
        if let bmi = bmi {
            guard bmi >= 10 && bmi <= 80 else {
                throw ActuarialError.invalidHealthMetric("BMI must be between 10-80")
            }
        }
        
        if let vo2 = vo2Max {
            guard vo2 >= 10 && vo2 <= 90 else {
                throw ActuarialError.invalidHealthMetric("VO2 Max must be between 10-90 ml/kg/min")
            }
        }
        
        if let sleep = sleepHours {
            guard sleep >= 0 && sleep <= 24 else {
                throw ActuarialError.invalidHealthMetric("Sleep hours must be between 0-24")
            }
        }
    }
}